<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\Validator;

class PresensiMahasiswa extends Model
{
    use HasFactory; // Menggunakan trait HasFactory untuk factory model

    protected $table = 't_absen';

    // Definisikan kolom yang dapat diisi secara massal (mass assignable)
    protected $fillable = [
        'kd_jurusan', 'kd_tahun_ajar', 'kd_semester', 'kd_kelas', 'kd_matakuliah', 'tanggal', 'pertemuan', 'nim', 'kd_absen'
    ];

    // Definisikan relasi dengan model lainnya
    public function jurusan()
    {
        return $this->belongsTo(Jurusan::class, 'kd_jurusan', 'kd_jurusan');
    }

    public function tahunAjar()
    {
        return $this->belongsTo(TahunAjar::class, 'kd_tahun_ajar', 'kd_tahun_ajar');
    }

    public function semester()
    {
        return $this->belongsTo(Semester::class, 'kd_semester', 'kd_semester');
    }

    public function kelas()
    {
        return $this->belongsTo(Kelas::class, 'kd_kelas', 'kode_kelas');
    }

    public function mahasiswa()
    {
        return $this->belongsTo(Mahasiswa::class, 'nim', 'nim');
    }

    // Metode untuk menambahkan presensi mahasiswa dengan validasi
    public function addPresensiMahasiswa(array $data)
    {
        $validator = Validator::make($data, [
            'kd_jurusan' => 'required',
            'kd_tahun_ajar' => 'required',
            'kd_semester' => 'required',
            'kd_kelas' => 'required',
            'kd_matakuliah' => 'required',
            'tanggal' => 'required',
            'pertemuan' => 'required',
            'nim' => 'required',
            'kd_absen' => 'required',
        ]);

        if ($validator->fails()) {
            throw new \Illuminate\Validation\ValidationException($validator);
        }

        return $this->create($data);
    }

    // Metode untuk memperbarui presensi mahasiswa
    public function updatePresensiMahasiswa($id, array $data)
    {
        $validator = Validator::make($data, [
            'kd_jurusan' => 'required',
            'kd_tahun_ajar' => 'required',
            'kd_semester' => 'required',
            'kd_kelas' => 'required',
            'kd_matakuliah' => 'required',
            'tanggal' => 'required',
            'pertemuan' => 'required',
            'nim' => 'required',
            'kd_absen' => 'required',
        ]);

        if ($validator->fails()) {
            throw new \Illuminate\Validation\ValidationException($validator);
        }

        // Temukan record berdasarkan ID dan perbarui
        $presensi = $this->find($id);
        if (!$presensi) {
            throw new \Exception('Record tidak ditemukan.');
        }

        $presensi->update($data);
        return $presensi;
    }
    
    // Metode untuk mendapatkan data presensi mahasiswa berdasarkan NIM dan tanggal
    public function getPresensiMahasiswa($nim, $kd_matakuliah, $kd_jurusan, $kd_tahun_ajar, $kd_semester, $kd_kelas, $pertemuan)
    {
        return $this->where('nim', $nim)
                    ->where('kd_matakuliah', $kd_matakuliah)
                    ->where('kd_jurusan', $kd_jurusan)
                    ->where('kd_tahun_ajar', $kd_tahun_ajar)
                    ->where('kd_semester', $kd_semester)
                    ->where('kd_kelas', $kd_kelas)
                    ->where('pertemuan', $pertemuan)
                    ->first();
    }

    // Metode untuk menghitung jumlah kehadiran mahasiswa
    public function countKehadiranMahasiswa($nim, $kd_matakuliah, $kd_jurusan, $kd_tahun_ajar, $kd_semester, $kd_kelas, $pertemuan)
    {
        return $this->where('nim', $nim)
                    ->where('kd_matakuliah', $kd_matakuliah)
                    ->where('kd_jurusan', $kd_jurusan)
                    ->where('kd_tahun_ajar', $kd_tahun_ajar)
                    ->where('kd_semester', $kd_semester)
                    ->where('kd_kelas', $kd_kelas)
                    ->where('pertemuan', $pertemuan)
                    ->count();
    }

    // Metode untuk menghapus presensi mahasiswa
    public function destroyPresensiMahasiswa($nim, $kd_matakuliah, $kd_jurusan, $kd_tahun_ajar, $kd_semester, $kd_kelas, $pertemuan)
    {
        // Mencari record berdasarkan kondisi yang diberikan
        $presensi = $this->where('nim', $nim)
                         ->where('kd_matakuliah', $kd_matakuliah)
                         ->where('kd_jurusan', $kd_jurusan)
                         ->where('kd_tahun_ajar', $kd_tahun_ajar)
                         ->where('kd_semester', $kd_semester)
                         ->where('kd_kelas', $kd_kelas)
                         ->where('pertemuan', $pertemuan)
                         ->first();

        if (!$presensi) {
            throw new \Exception('Record tidak ditemukan.');
        }

        return $presensi->delete();
    }
}
