<?php

namespace App\Http\Controllers;

use App\Models\Notif;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Validator;
use Illuminate\Support\Facades\DB;

class NotifController extends Controller
{
    public function __construct()
    {
        // Hanya authenticated user yang bisa mengakses beberapa fungsi
        $this->middleware('auth', ['only' => ['store', 'update', 'destroy', 'index', 'show']]);
    }

    public function index(Request $request)
    {
        // Ambil semua notifikasi
        $notif = Notif::all();

        return response()->json($notif);
    }
    
    public function show($id)
    {
        // Mengambil semua notifikasi dengan ID yang sama
        $notifs = Notif::where('nim', $id)->get();
        
        // Jika tidak ada notifikasi ditemukan, kembalikan error 404
        if ($notifs->isEmpty()) {
            return response()->json(['message' => 'Data not found'], 404);
        }
        
        // Kembalikan semua notifikasi yang ditemukan
        return response()->json($notifs);
    }


    public function store(Request $request)
    {
        // Log request input
        \Log::info('Request Data:', $request->all());
    
        // Validasi data array notifikasi
        $validator = Validator::make($request->all(), [
            'data.*.id_user' => 'required|integer',
            'data.*.nim' => 'required|string',
            'data.*.deskripsi' => 'required|string',
        ]);
    
        // Jika validasi gagal
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()->all()], 422);
        }
    
        // Mulai transaksi
        DB::beginTransaction();
        try {
            // Ambil array data dari request
            $dataArray = $request->input('data');
    
            // Cek apakah $dataArray adalah array yang valid
            if (!is_array($dataArray) || empty($dataArray)) {
                return response()->json(['error' => 'Invalid input data'], 400);
            }
    
            // Simpan setiap notifikasi dalam array
            $notifications = [];
            foreach ($dataArray as $dataInput) {
                // Tambahkan status 0
                $dataInput['status'] = 0; // Tetapkan status menjadi 0
                $notifications[] = Notif::create($dataInput);
            }
    
            // Commit transaksi jika semua sukses
            DB::commit();
    
            // Kembalikan response dengan data notifikasi yang tersimpan
            return response()->json(['notifications' => $notifications], 201);
        
        } catch (\Exception $e) {
            // Rollback transaksi jika ada error
            DB::rollBack();
    
            // Logging kesalahan
            \Log::error('Failed to save notifications: ' . $e->getMessage());
    
            // Kembalikan pesan error
            return response()->json(['error' => 'Failed to save notifications', 'message' => $e->getMessage()], 500);
        }
    }



}
